package edu.uky.ai.lp;

/**
 * Describes the result of a single action taken by the player, including any
 * new information gained.
 * 
 * @author Stephen G. Ware
 */
public class Result {

	/** Indicates if the square the player just moved to has a stench */
	public final boolean stench;
	
	/** Indicates if the square the player just moved to has a breeze */
	public final boolean breeze;
	
	/** Indicates if the square the player just moved to glitters */
	public final boolean glitter;
	
	/** A specific message to be displayed as a result of special actions */
	private final String message;
	
	/**
	 * Creates a result which indicates sensory information about the square
	 * the player just moved to.
	 * 
	 * @param stench indicates if the square the player just moved to has a stench 
	 * @param breeze indicates if the square the player just moved to has a breeze 
	 * @param glitter indicates if the square the player just moved to glitters 
	 */
	public Result(boolean stench, boolean breeze, boolean glitter) {
		this.stench = stench;
		this.breeze = breeze;
		this.glitter = glitter;
		this.message = null;
	}
	
	/**
	 * Creates a result which displayed a special message.
	 * 
	 * @param message the message to display
	 */
	public Result(String message) {
		this.stench = false;
		this.breeze = false;
		this.glitter = false;
		this.message = message;
	}
	
	@Override
	public String toString() {
		if(message != null)
			return message;
		String str = "";
		if(stench)
			str += "You smell a wumpus.\n";
		if(breeze)
			str += "You feel a breeze.\n";
		if(glitter)
			str += "You see a glittering.\n";
		if(str.length() == 0)
			str = "You sense nothing.";
		else
			str = str.substring(0, str.length() - 1);
		return str;
	}
	
	/** A special result indicating that the player's action has no effect on the game state */
	public static final Result NOTHING = new Result("Nothing happens...");
	
	/** A special result indicating that the player picked up the gold */
	public static final Result GRAB = new Result("You grab the gold!");
	
	/** A special result indicating that the player won */
	public static final Result WIN = new Result("You have won!");
	
	/** A special result indicating that the player was eaten by the wumpus */
	public static final Result EAT = new Result("You are eaten by the wumpus!");
	
	/** A special result indicating that the player fell into a pit */
	public static final Result FALL = new Result("You fall into a bottomless pit!");
}
